﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Data;
using System.Windows.Forms;
using System.Xml;
using System.Text.RegularExpressions;

namespace Manuels_NotenChecker.Domain
{
    class NotenChecker
    {
        private class NotenXmlGenerator
        {
            private System.IO.MemoryStream memStream;
            private XmlWriter writer;
            private const string NOTENSPIEGEL_NAMESPACE = "http://neokc.de/NotenChecker/Notenspiegel.xsd";

            public NotenXmlGenerator()
            {
                this.memStream = new System.IO.MemoryStream();
                XmlWriterSettings settings = new XmlWriterSettings();
                settings.Encoding = new UTF8Encoding(false);
                settings.ConformanceLevel = ConformanceLevel.Document;
                settings.Indent = true;
                
                this.writer = XmlTextWriter.Create(this.memStream, settings);

                writer.WriteStartDocument();
                writer.WriteStartElement("nsp", "noten", NOTENSPIEGEL_NAMESPACE);
                writer.WriteAttributeString("xmlns", "nsp", null, NOTENSPIEGEL_NAMESPACE);
            }

            public void AppendRecord(string prnr, bool status, string pruefungstext, string semester, DateTime pruefungsdatum, string note, string vermerk, int versuch)
            {
                string statusString = status.ToString().ToLower();
                string pruefungsdatumString = pruefungsdatum.ToString("yyyy-MM-dd");
                string versuchString = versuch.ToString();

                this.writer.WriteStartElement("note", NOTENSPIEGEL_NAMESPACE);

                this.writer.WriteAttributeString("PrNr", prnr);
                this.writer.WriteAttributeString("Status", statusString);
                this.writer.WriteElementString("Pruefungstext", NOTENSPIEGEL_NAMESPACE, pruefungstext);
                this.writer.WriteElementString("Semester", NOTENSPIEGEL_NAMESPACE, semester);
                this.writer.WriteElementString("Pruefungsdatum", NOTENSPIEGEL_NAMESPACE, pruefungsdatumString);
                this.writer.WriteElementString("Note", NOTENSPIEGEL_NAMESPACE, note);
                this.writer.WriteElementString("Vermerk", NOTENSPIEGEL_NAMESPACE, vermerk);
                this.writer.WriteElementString("Versuch", NOTENSPIEGEL_NAMESPACE, versuchString);
                
                writer.WriteEndElement();
            }

            public string GetCompleteXml()
            {
                this.writer.WriteEndElement();
                this.writer.Flush();
                this.writer.Close();
                
                string xmlInhalt = Encoding.UTF8.GetString(this.memStream.ToArray());
                return xmlInhalt;
            }
        }

        public enum checkStates
        {
            NOTHING,
            CONNECT,
            LOGIN,
            LOGIN_FAILED,
            PARSE_MARKS,
            DISCONNECT
        }

        private const string LOGIN_PAGE = "https://qis2.hs-karlsruhe.de/qisserver/rds?state=user&type=0&topitem=&breadCrumbSource=portal&topitem=functions";
        private const string LOGOUT_PAGE = "https://qis2.hs-karlsruhe.de/qisserver/rds?state=user&amp;type=4&amp;category=auth.logout&amp;menuid=logout&amp;breadcrumb=Logout&amp;breadCrumbSource=loggedin";

        public delegate void CheckCompleted(string xmlString);
        public delegate void StateChanged(checkStates state);
        public delegate void ErrorHandler(string errorText);

        private CheckCompleted completed;
        private StateChanged statech;
        private ErrorHandler errorH;
        private string user;
        private string pass;
        private WebBrowser webBr;
        private checkStates currentState;

        public NotenChecker(CheckCompleted completed, StateChanged statech, ErrorHandler errorH)
        {
            this.completed = completed;
            this.statech = statech;
            this.errorH = errorH;

            this.webBr = new System.Windows.Forms.WebBrowser();
            //this.webBr.WebBrowserShortcutsEnabled = false;
            this.webBr.ScriptErrorsSuppressed = true;

            this.webBr.DocumentCompleted += 
                new WebBrowserDocumentCompletedEventHandler(this.webBr_DocumentCompleted);
            this.ChangeState(checkStates.NOTHING);
        }

        public string User
        {
            get
            {
                return this.user;
            }
            set
            {
                this.user = value;
            }
        }

        public string Pass
        {
            set
            {
                this.pass = value;
            }
        }

        public void Check()
        {
            this.ChangeState(checkStates.CONNECT);
            this.webBr.Navigate(LOGIN_PAGE);
        }

        private static string GetLinkOfLabel(string source, string label)
        {
            MatchCollection nslink = Regex.Matches(source, "(?<=>( *)<(a|A)(.*)(href|HREF)=\")([^\"]*)(?=\">" + label + ")");
            return nslink[0].Value.Replace("&amp;", "&");
        }

        private void DocumentStateConnect(HtmlElement bodyEl)
        {
            try
            {   
                HtmlElementCollection inputs = bodyEl.GetElementsByTagName("input");
                if (inputs.Count >= 3)
                {
                    HtmlElement user = inputs[0];
                    HtmlElement pass = inputs[1];
                    HtmlElement submit = inputs[2];
                    user.SetAttribute("value", this.user);
                    pass.SetAttribute("value", this.pass);
                    this.ChangeState(checkStates.LOGIN);
                    submit.InvokeMember("Click");
                }
                else
                {
                    throw new Exception("Login Felder konnten nicht ermittelt werden.");
                }
            }
            catch (Exception)
            {
                throw new Exception("Verbindung zum Hochschuldienst konnte nicht aufgebaut werden.");
            }
        }

        private void DocumentStateDisconnect(HtmlElement bodyEl)
        {
            this.ChangeState(checkStates.NOTHING);
        }

        private void DocumentStateLogin(HtmlElement bodyEl)
        {
            if (bodyEl.InnerText.Contains("Anmeldung fehlgeschlagen"))
            {
                this.ChangeState(checkStates.LOGIN_FAILED);
            }
            else
            {
                DocumentStateGetmarks(bodyEl);
            }
        }
        
        private void DocumentStateGetmarks(HtmlElement bodyEl)
        {
            try
            {
                if (!bodyEl.InnerHtml.Contains("Notenspiegel"))
                {
                    string nsUrl = GetLinkOfLabel(bodyEl.InnerHtml, "Prüfungsverwaltung");
                    this.webBr.Navigate(nsUrl);
                    return;
                }

                if (!bodyEl.InnerHtml.Contains("Abschluss "))
                {
                    string nsUrl = GetLinkOfLabel(bodyEl.InnerHtml, "Notenspiegel");
                    this.webBr.Navigate(nsUrl);
                    return;
                }

                if (bodyEl.InnerHtml.Contains("Abschluss "))
                {
                    string nsUrl = GetLinkOfLabel(bodyEl.InnerHtml, "<IMG title=\"Leistungen für ");
                    this.ChangeState(checkStates.PARSE_MARKS);
                    this.webBr.Navigate(nsUrl);
                    return;
                }

                throw new Exception();
            }
            catch (Exception)
            {
                throw new Exception("Notenspiegel konnte nicht gefunden werden.");
            }
        }

        private void DocumentStateParsemarks(HtmlElement bodyEl)
        {
            try
            {
                NotenXmlGenerator xmlGen = new NotenXmlGenerator();
                MatchCollection matches = Regex.Matches(bodyEl.InnerHtml, "(?<=<TD class=tabelle1_.*>)(.*)(?=</TD>)");
                for (int i = 0; i < matches.Count; i += 8)
                {
                    string prnr = matches[i + 0].Value.Trim();
                    string pruefungstext = matches[i + 1].Value.Trim();
                    string semester = matches[i + 2].Value.Trim();
                    string pruefungsdatum = matches[i + 3].Value.Trim();
                    DateTime pruefungsDateTime = DateTime.ParseExact(pruefungsdatum, "dd.MM.yyyy", null);
                    string note = matches[i + 4].Value.Split(new char[]{' '})[0].Trim();
                    string status = matches[i + 5].Value.Trim();
                    bool statusBool = status == "bestanden";
                    string vermerk = matches[i + 6].Value.Trim();
                    string versuch = matches[i + 7].Value.Trim();
                    int veruchInt = Int32.Parse(versuch);
                    xmlGen.AppendRecord(prnr, statusBool, pruefungstext, semester, pruefungsDateTime, note, vermerk, veruchInt);
                }
                this.completed(xmlGen.GetCompleteXml());
                this.DisconnectService();
            }
            catch (Exception)
            {
                throw new Exception("Noten konnten nicht ausgelesen werden.");
            }
        }

        private void DisconnectService()
        {
            this.ChangeState(checkStates.DISCONNECT);
            this.webBr.Navigate(LOGOUT_PAGE);
        }

        private void DocumentLoadedCompletly()
        {
            HtmlElement bodyEl = this.webBr.Document.Body.Document.Body;

            switch (this.currentState)
            {
                case checkStates.CONNECT:
                    {
                        this.DocumentStateConnect(bodyEl);
                        break;
                    }
                case checkStates.LOGIN:
                    {
                        this.DocumentStateLogin(bodyEl);
                        break;
                    }
                case checkStates.DISCONNECT:
                    {
                        this.DocumentStateDisconnect(bodyEl);
                        break;
                    }
                case checkStates.PARSE_MARKS:
                    {
                        this.DocumentStateParsemarks(bodyEl);
                        break;
                    }
                default:
                    {
                        throw new Exception("Ungültige Operation.");
                    }
            }

        }

        private void webBr_DocumentCompleted(object sender, WebBrowserDocumentCompletedEventArgs e)
        {
            try
            {
                this.DocumentLoadedCompletly();
            }
            catch (Exception ex)
            {
                this.errorH(ex.Message);
                this.DisconnectService();
            }
        }

        private void ChangeState(checkStates state)
        {
            this.currentState = state;
            this.statech(state);
        }
    }
}
